<?php
// worker_search.php
// CLI worker: جستجوی حساب کاربر در سرور ها و ارسال کیبورد انتخاب مدت
// Usage: php worker_search.php <order_id>

if (PHP_SAPI !== 'cli') {
    // اجازه می دهیم از وب هم اجرا شود ولی توصیه شده CLI باشد
}

/* -------------------------
   bootstrap
   ------------------------- */
if (!isset($argv) || count($argv) < 2) {
    echo "Usage: php worker_search.php <order_id>\n";
    exit(1);
}
$order_id = intval($argv[1]);
if ($order_id <= 0) {
    echo "order_id نامعتبر است\n";
    exit(1);
}

require __DIR__ . '/functions.php';
$config = require __DIR__ . '/config.php';
$conn = db_connect();

/* -------------------------
   read temp input created by webhook
   ------------------------- */
$tmpf = __DIR__ . "/tmp_order_search_{$order_id}.json";
if (!file_exists($tmpf)) {
    exit(0);
}

$raw_json = @file_get_contents($tmpf);
$raw = @json_decode($raw_json, true);
if (!$raw) {
    // دادهٔ نامعتبر — حذف فایل و خروج
    @unlink($tmpf);
    exit(0);
}

$chat_id = intval($raw['chat_id'] ?? 0);
$text = $raw['text'] ?? '';
$parsed = $raw['parsed'] ?? null;
if (!$chat_id) {
    @unlink($tmpf);
    exit(0);
}

$identity = is_array($parsed) ? $parsed : extract_identity_from_link($text);
if (!$identity || !isset($identity['type'])) {
    // اطلاعات هویت ناقص است
    @unlink($tmpf);
    exit(0);
}

/* -------------------------
   iterate over panels and search
   ------------------------- */
$found = null;

$res = mysqli_query($conn, "SELECT * FROM services_panels");
if ($res && mysqli_num_rows($res) > 0) {
    while ($row = mysqli_fetch_assoc($res)) {
        $panel_id = intval($row['id'] ?? 0);

        // try login (panel_login may be called inside helpers as well)
        try {
            if (function_exists('panel_login')) {
                $ok = false;
                try {
                    $ok = panel_login($row);
                    panel_debug_log($panel_id, "Initial panel_login returned: " . ($ok ? 'true' : 'false'));
                } catch (Throwable $e) {
                    panel_debug_log($panel_id, "panel_login() threw: " . $e->getMessage());
                }
            }
        } catch (Throwable $e) {
            // ignore
        }

        // fetch inbounds list (helper will login if needed)
        try {
            if (function_exists('panel_get_inbounds_list')) {
                $inbounds_list = panel_get_inbounds_list($row);
                panel_debug_log($panel_id, "panel_get_inbounds_list returned count=" . (is_array($inbounds_list) ? count($inbounds_list) : 'null'));
            }
        } catch (Throwable $e) {
            panel_debug_log($panel_id, "panel_get_inbounds_list() threw: " . $e->getMessage());
            $inbounds_list = null;
        }

        // try to find account
        try {
            if (function_exists('panel_find_account_via_inbounds')) {
                $try = panel_find_account_via_inbounds($row, $identity);
                if ($try && is_array($try)) {
                    $found = $try;
                    panel_debug_log($panel_id, "Found match: matched_by={$try['matched_by']} matched_field=" . substr(var_export($try['matched_field'], true), 0, 200));
                    break;
                } else {
                    panel_debug_log($panel_id, "No match found on this panel");
                }
            }
        } catch (Throwable $e) {
            panel_debug_log($panel_id, "panel_find_account_via_inbounds() threw: " . $e->getMessage());
        }
    }
}

/* -------------------------
   handle result
   ------------------------- */
if ($found) {
    $panel = $found['panel'];
    $inbound = $found['inbound'];
    $client = $found['client'];

    $panel_id = intval($panel['id'] ?? 0);
    // client id can be string (uuid) or numeric
    $panel_account_id = isset($client['id']) ? (string)$client['id'] : (isset($client['uuid']) ? (string)$client['uuid'] : (isset($client['subId']) ? (string)$client['subId'] : ''));

    // Update orders: set panel_id, panel_account_id, status = 'pending'
    $stmt = mysqli_prepare($conn, "UPDATE orders SET panel_id = ?, panel_account_id = ?, status = 'pending' WHERE id = ?");
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, 'isi', $panel_id, $panel_account_id, $order_id);
        @mysqli_stmt_execute($stmt);
    }

    // fetch distinct durations (duration_days) allowed for this panel from services_plans
    $durations = [];
    $dur_stmt = mysqli_prepare($conn, "
        SELECT DISTINCT p.duration_days
        FROM services_plans p
        JOIN services_relation_plans r ON p.id = r.related_plan_id
        WHERE r.related_panel_id = ?
        ORDER BY p.duration_days ASC
    ");
    if ($dur_stmt) {
        mysqli_stmt_bind_param($dur_stmt, 'i', $panel_id);
        mysqli_stmt_execute($dur_stmt);
        $dres = mysqli_stmt_get_result($dur_stmt);
        if ($dres) {
            $durations = mysqli_fetch_all($dres, MYSQLI_ASSOC);
        }
    }

    // build inline keyboard: one button per duration (فارسی + emoji)
    $keyboard = ['inline_keyboard' => []];
    if (!empty($durations)) {
        foreach ($durations as $d) {
            $days = intval($d['duration_days'] ?? 0);
            if ($days <= 0) continue;

            // محاسبه برچسب خوانا براساس روز
            if ($days < 1) {
                // theoretical — but duration_days is integer so skip
                $duration_label = "{$days} روز";
            } elseif ($days < 30) {
                $duration_label = "{$days} روز";
            } else {
                $months = intdiv($days, 30);
                if ($months < 12) {
                    $duration_label = ($months <= 1) ? "1 ماه" : "{$months} ماه";
                } else {
                    $years = intdiv($months, 12);
                    $duration_label = ($years <= 1) ? "1 سال" : "{$years} سال";
                }
            }

            $label = "🔹 {$duration_label}";
            // callback_data uses duration_days directly
            $keyboard['inline_keyboard'][] = [['text' => $label, 'callback_data' => "pick_d:{$order_id}:{$days}"]];
        }
    } else {
        // No durations for this panel: provide cancel button and notify admins (admins will apply manually)
        $keyboard['inline_keyboard'][] = [['text' => '✖️ انصراف', 'callback_data' => "cancel_order:{$order_id}"]];
    }

    // always add cancel row if not already present
    $keyboard['inline_keyboard'][] = [['text' => '✖️ انصراف', 'callback_data' => "cancel_order:{$order_id}"]];

    $panel_name = $panel['server_name'] ?? $panel['url'] ?? 'سرور ناشناس';
    $total_gb = $used_gb = $remain_gb = 0;
    $remain_time = "—";
    $endpoint = '';
    $resp_preview = '';

    try {
        $email_id = $client['email'] ?? null;
        if ($email_id && !empty($panel['url'])) {
            // Use the helper - it will ensure login cookie exists and retry
            $traffics = null;
            try {
                if (function_exists('panel_get_client_traffics')) {
                    $traffics = panel_get_client_traffics($panel, $email_id);
                    panel_debug_log($panel_id, "panel_get_client_traffics returned " . (is_array($traffics) ? 'array' : 'null'));
                } else {
                    panel_debug_log($panel_id, "panel_get_client_traffics() not defined");
                }
            } catch (Throwable $e) {
                panel_debug_log($panel_id, "panel_get_client_traffics() threw: " . $e->getMessage());
                $traffics = null;
            }

            // For message display, build endpoint preview and resp_preview
            $urlBase = rtrim($panel['url'], '/');
            $endpoint = "{$urlBase}/panel/api/inbounds/getClientTraffics/{$email_id}";

            if (is_array($traffics)) {
                // prefer structure that matches earlier inline expectation: { success:true, obj:[ ... ] }
                if (isset($traffics['obj'])) {
                    // Handle both array and object
                    $ci = is_array($traffics['obj']) && isset($traffics['obj'][0])
                        ? $traffics['obj'][0]
                        : $traffics['obj'];
                    $total_gb = round(($ci['total'] ?? 0) / (1024 ** 3), 2);
                    $used_gb  = round((($ci['up'] ?? 0) + ($ci['down'] ?? 0)) / (1024 ** 3), 2);
                    $remain_gb = max(0, $total_gb - $used_gb);

                    $expiry = intval($ci['expiryTime'] ?? 0);
                    if ($expiry > 0) {
                        $exp_ts = $expiry / 1000;
                        $diff = $exp_ts - time();
                        if ($diff > 0) {
                            if ($diff < 3600) {
                                $remain_time = floor($diff / 60) . " دقیقه";
                            } elseif ($diff < 86400) {
                                $remain_time = floor($diff / 3600) . " ساعت";
                            } elseif ($diff < 2592000) {
                                $remain_time = floor($diff / 86400) . " روز";
                            } elseif ($diff < 31104000) {
                                $remain_time = floor($diff / 2592000) . " ماه";
                            } else {
                                $remain_time = "بیش از یک سال";
                            }
                        } else {
                            $remain_time = "منقضی شده";
                        }
                    }
                } else {
                    // decoded but missing obj[0] — keep preview
                    $resp_preview = substr(json_encode($traffics, JSON_UNESCAPED_UNICODE), 0, 800);
                    panel_debug_log($panel_id, "Traffic response decoded but missing obj[0] — preview saved");
                }
            } else {
                panel_debug_log($panel_id, "panel_get_client_traffics returned null or non-array for email {$email_id}");
            }
        } else {
            panel_debug_log($panel_id, "No email_id available for client or panel.url missing; skipping traffic API");
        }
    } catch (Throwable $e) {
        panel_debug_log($panel_id, "Traffic API exception: " . $e->getMessage());
    }

    // build display message
    $panel_name = $panel['server_name'] ?? 'سرور ناشناس';
    $msg =
           "✅ حساب شما در سرور «{$panel_name}» پیدا شد.\n"
         . "📊 حجم کل: {$total_gb} گیگابایت\n"
         . "📈 مصرف شده: {$used_gb} گیگابایت\n"
         . "📉 باقی مانده: {$remain_gb} گیگابایت\n"
         . "⏰ زمان باقی مانده: {$remain_time}\n\n"
         . "لطفاً مدت مورد نظر را انتخاب کنید:";

    try {
        telegram_send_message($chat_id, $msg, $keyboard);
    } catch (Throwable $e) {
        // best-effort: if sending پیام fails, notify admins with plain text
        try {
            telegram_send_to_admins("⚠️ خطا در ارسال پیام برای کاربر {$chat_id} در سفارش {$order_id}. لطفاً دستی بررسی کنید.");
        } catch (Throwable $ignore) {}
    }

    // remove temp file
    @unlink($tmpf);
    exit(0);
} else {
    // Not found: update order status to not_found and inform user & admins (Persian)
    $stmt = mysqli_prepare($conn, "UPDATE orders SET status='not_found' WHERE id=?");
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, 'i', $order_id);
        @mysqli_stmt_execute($stmt);
    }

    // notify user
    try {
        telegram_send_message($chat_id, "❌ متأسفم، حساب شما در هیچ یک از سرور های پیکربندی شده پیدا نشد. مدیران مطلع شدند و در صورت نیاز با شما تماس خواهند گرفت.");
    } catch (Throwable $e) {
        // ignore
    }

    // notify admins with order info (no inline API response logging)
    $preview_text = substr($text ?? '', 0, 400);
    $admin_text = "🔎 جستجو برای سفارش {$order_id} ناموفق بود.\nکاربر: {$chat_id}\nمتن ارسالی: {$preview_text}\nلطفاً بررسی دستی انجام دهید.";
    try {
        telegram_send_to_admins($admin_text);
    } catch (Throwable $e) {
        // ignore
    }

    // remove temp file
    @unlink($tmpf);
    exit(0);
}
