<?php
// pay_redirect/index.php
// Serve over HTTPS. This script ensures the current host matches the registered site_url host
// and then shows a short invoice/confirmation page that redirects user to Zarinpal StartPay.
// It expects config.php (or configs/config.php) to return an array with at least:
//   - site_url (e.g. "https://example.com")
//   - z_sign_key
// Optional config keys used:
//   - zarinpal_sandbox (bool)
//   - z_redirect_require_db (bool)
//   - z_redirect_force_header (bool)   // if true, uses a 302 header redirect (legacy)

// load config
$configPath1 = __DIR__ . '/../config.php';
$configPath2 = __DIR__ . '/../configs/config.php';
$config = null;
if (file_exists($configPath1)) {
    $config = include $configPath1;
} elseif (file_exists($configPath2)) {
    $config = include $configPath2;
}

if (!$config || empty($config['site_url']) || empty($config['z_sign_key'])) {
    http_response_code(500);
    echo "Server configuration error.";
    exit;
}

// --- helper: normalize host from a URL or plain host ---
function host_from_url_or_host($u) {
    $u = trim($u);
    if ($u === '') return '';
    // if no scheme, add http:// to let parse_url parse host
    if (!preg_match('#^[a-z]+://#i', $u)) {
        $u = 'https://' . $u;
    }
    $h = parse_url($u, PHP_URL_HOST);
    if (!$h) $h = $u;
    // normalize: lowercase and strip leading www.
    $h = strtolower($h);
    if (strpos($h, 'www.') === 0) $h = substr($h, 4);
    return $h;
}

// ensure HTTPS (we want the browser to send Referer; also security)
$isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ||
           (isset($_SERVER['SERVER_PORT']) && ($_SERVER['SERVER_PORT'] == 443)) ||
           (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https');

if (!$isHttps) {
    http_response_code(403);
    echo "This endpoint must be served over HTTPS.";
    exit;
}

// current host
$currentHost = $_SERVER['HTTP_HOST'] ?? ($_SERVER['SERVER_NAME'] ?? '');
$currentHost = strtolower($currentHost);
if (strpos($currentHost, 'www.') === 0) $currentHost = substr($currentHost, 4);

// host from config site_url
$configHost = host_from_url_or_host($config['site_url'] ?? '');

// require hosts to match
if (!$configHost || $currentHost !== $configHost) {
    http_response_code(400);
    // friendly HTML explaining why (so integrator knows how to fix)
    echo "<!doctype html><html lang='fa'><head><meta charset='utf-8'><title>Domain mismatch</title></head><body style='font-family:system-ui, -apple-system, \"Segoe UI\", Roboto, \"Helvetica Neue\", Arial;direction:rtl; padding:20px;'>";
    echo "<h2>نقص در دامنهٔ ارجاع‌دهنده</h2>";
    echo "<p>این صفحه باید روی همان دامنه‌ای میزبانی شود که در زرین‌پال ثبت شده است.</p>";
    echo "<ul>";
    echo "<li>دامنهٔ کنونی: <strong>" . htmlspecialchars($currentHost) . "</strong></li>";
    echo "<li>دامنهٔ ثبت‌شده در کانفیگ: <strong>" . htmlspecialchars($configHost) . "</strong></li>";
    echo "</ul>";
    echo "<p>برای رفع:</p>";
    echo "<ol>";
    echo "<li>این فایل را روی دامنهٔ رسمی (مثلاً <code>" . htmlspecialchars($configHost) . "</code>) منتشر کنید، یا</li>";
    echo "<li>یک هدایت‌گر (forwarder) از دامنهٔ رسمی به این مسیر راه‌اندازی کنید تا مرورگر ارجاع‌دهندهٔ صحیح ارسال گردد.</li>";
    echo "</ol>";
    echo "</body></html>";
    exit;
}

// read input
$authority = $_GET['auth'] ?? '';
$order_id  = $_GET['order'] ?? '';
$ts        = isset($_GET['ts']) ? (int)$_GET['ts'] : 0;
$sig       = $_GET['sig'] ?? '';

if (!$authority || !$order_id || !$ts || !$sig) {
    http_response_code(400);
    echo "Bad request.";
    exit;
}

// validate timestamp window (prevent replay)
$window = 60 * 60; // 1 hour
if (abs(time() - $ts) > $window) {
    http_response_code(410);
    echo "Link expired.";
    exit;
}

// validate signature
$expected = hash_hmac('sha256', "{$authority}|{$order_id}|{$ts}", $config['z_sign_key']);
if (!hash_equals($expected, $sig)) {
    http_response_code(403);
    echo "Invalid signature.";
    exit;
}

// Optional DB check
$require_db = isset($config['z_redirect_require_db']) ? (bool)$config['z_redirect_require_db'] : false;
$found = true;

if ($require_db) {
    $dbHost = $config['db_host'] ?? null;
    $dbUser = $config['db_user'] ?? null;
    $dbPass = $config['db_pass'] ?? null;
    $dbName = $config['db_name'] ?? null;
    if (!$dbHost || !$dbUser || !$dbName) {
        http_response_code(500);
        echo "DB config missing.";
        exit;
    }

    $mysqli = @new mysqli($dbHost, $dbUser, $dbPass, $dbName);
    if ($mysqli->connect_errno) {
        http_response_code(500);
        echo "DB connection failed.";
        exit;
    }

    $stmt = $mysqli->prepare("SELECT status FROM payments WHERE order_id = ? AND provider_ref = ? LIMIT 1");
    if ($stmt) {
        $order_int = (int)$order_id;
        $stmt->bind_param('is', $order_int, $authority);
        $stmt->execute();
        $stmt->bind_result($status);
        if (!$stmt->fetch()) {
            $found = false;
        }
        $stmt->close();
    } else {
        $found = false;
    }
    $mysqli->close();

    if (!$found) {
        http_response_code(404);
        echo "Payment not found.";
        exit;
    }
}

// Decide zarinpal host
$zHost = (!empty($config['zarinpal_sandbox']) ? 'https://sandbox.zarinpal.com' : 'https://www.zarinpal.com');
$target = "{$zHost}/pg/StartPay/" . rawurlencode($authority);

// Set safe headers
header('Referrer-Policy: origin'); // ensure domain (origin) is sent as Referer
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: SAMEORIGIN'); // avoid clickjacking

// Legacy: if config explicitly forces immediate header redirect, do it (but still we verified host + signature)
if (!empty($config['z_redirect_force_header'])) {
    header("Location: {$target}", true, 302);
    exit;
}

// Otherwise present a small invoice/confirm page and redirect via browser (preserves Referer).
$safeOrder = htmlspecialchars($order_id, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
$safeAuth  = htmlspecialchars($authority, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
$safeTarget = htmlspecialchars($target, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
?>
<!doctype html>
<html lang="fa" dir="rtl">
<head>
  <meta charset="utf-8">
  <title>هدایت به درگاه پرداخت</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body{font-family:system-ui, -apple-system, "Segoe UI", Roboto, "Helvetica Neue", Arial; background:#f7fafc; color:#111; padding:24px;}
    .card{max-width:720px;margin:32px auto;background:#fff;border-radius:12px;padding:20px; box-shadow:0 6px 20px rgba(0,0,0,0.06);}
    .btn{display:inline-block;padding:12px 18px;border-radius:8px;text-decoration:none;font-weight:600;border:1px solid #2b6cb0;background:#2b6cb0;color:#fff}
    p.small{color:#666;font-size:0.95rem}
  </style>
  <meta http-equiv="refresh" content="10;url=<?php echo $safeTarget; ?>">
</head>
<body>
  <div class="card" role="main" aria-labelledby="title">
    <h1 id="title">در حال آماده‌سازی پرداخت</h1>
    <p>شماره سفارش: <strong><?php echo $safeOrder; ?></strong></p>
    <p class="small">در چند ثانیه به درگاه پرداخت منتقل می‌شوید. در صورت تأخیر یا تمایل، از دکمهٔ زیر استفاده کنید.</p>

    <p style="margin-top:20px;">
      <a class="btn" id="payButton" href="<?php echo $safeTarget; ?>">رفتن به صفحهٔ پرداخت</a>
    </p>

    <p class="small" style="margin-top:16px;">
      توجه: برای جلوگیری از خطا و تعلیق درگاه، این پرداخت باید از دامنهٔ ثبت‌شده (<strong><?php echo htmlspecialchars($configHost); ?></strong>) آغاز شده و به همان دامنه بازگردد.
    </p>
  </div>

  <script>
    // auto-redirect after 5s with user-visible countdown
    (function(){
      var countdown = 5;
      var btn = document.getElementById('payButton');
      var interval = setInterval(function(){
        if (countdown <= 0) {
          clearInterval(interval);
          // use location.href so browser sends Referer
          window.location.href = "<?php echo $safeTarget; ?>";
          return;
        }
        btn.textContent = 'رفتن به صفحهٔ پرداخت (' + countdown + 's)';
        countdown--;
      }, 1000);
    })();
  </script>
</body>
</html>
